<?php

namespace App\Livewire\Admin;
use Livewire\Component;
use Livewire\WithPagination;
use App\Livewire\Traits\AuthorizesRequests;

use Exception;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Pagination\LengthAwarePaginator;


class BackupComponent extends Component
{
    use WithPagination,AuthorizesRequests;
    public $perPage = 10;

    public function render()
    {
        if (!Gate::allows('view_backup') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        // Path to the backup folder
        $backupFolder = 'backups/'.config('app.name');
        $files = Storage::files($backupFolder);
        $backups = [];
        if (!empty($files)) {
            $backups = collect($files)->map(function ($file) {
                return [
                    'name' => basename($file),
                    'size' => Storage::size($file), // File size in bytes
                    'last_modified' => Storage::lastModified($file), // Timestamp of last modification
                    'path' => $file, // Full path for actions like download
                ];
            })->sortByDesc('last_modified');
            // Convert collection to Laravel's paginator
            $currentPage = LengthAwarePaginator::resolveCurrentPage();
            $collection = new LengthAwarePaginator(
                $backups->slice(($currentPage - 1) * $this->perPage, $this->perPage),
                count($backups),
                $this->perPage,
                $currentPage,
                ['path' => request()->url()]
            );
        }else {
            // If no files, return an empty collection with pagination
            $emptyCollection = collect(); // Empty collection
            $paginatedEmptyCollection = $emptyCollection->forPage(1, 10); // Paginate even if empty
        
            // Manually create paginator for the empty collection
            $collection = new LengthAwarePaginator(
                $paginatedEmptyCollection,
                0, // Total items are 0
                10, // Items per page
                1, // Page number
                ['path' => url()->current()] // Path for pagination links
            );
        }
        return view('admin.backup-component', compact('collection'));
    }
    public function create(){
        if (!Gate::allows('create_backup') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        try {
            Artisan::call('backup:run');
            return $this->dispatch('response', ['type' => 'success', 'message' => 'Backup process completed!']);
        } catch (Exception $e) {
            return $this->dispatch('response', ['type' => 'error', 'message' => $e->getMessage().' Something went wrong!']);
        }
    }
    public function download($file_name){
        if (!Gate::allows('view_backup') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        try{
            $backupFolder = 'backups/'.config('app.name');
            $filePath = $backupFolder . '/' . $file_name;
    
            if (Storage::exists($filePath)) {
                return Storage::download($filePath);
            }else{
                $this->dispatch('response', ['type' => 'error', 'message' => 'Backup file doesnt exist']);
            }
        } catch (Exception $e) {
            \Log::emergency("File:" . $e->getFile(). "Line:" . $e->getLine(). "Message:" . $e->getMessage());
            return $this->dispatch('response', ['type' => 'error', 'message' => $e->getMessage().' Something went wrong!']);
        }
    }
    public function delete($file_name){
        if (!Gate::allows('delete_backup') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        try{
            $backupFolder = 'backups/'.config('app.name');
            $filePath = $backupFolder . '/' . $file_name;
            if (Storage::exists($filePath)) {
                Storage::delete($filePath);
                return $this->dispatch('response', ['type' => 'error', 'message' => 'Database deleted successfully!']);
            }else{
                return $this->dispatch('response', ['type' => 'error', 'message' => 'Database file does not exist!']);
            }
        } catch (Exception $e) {
            \Log::emergency("File:" . $e->getFile(). "Line:" . $e->getLine(). "Message:" . $e->getMessage());
            return $this->dispatch('response', ['type' => 'error', 'message' => $e->getMessage().' Something went wrong!']);
        }
    }
    public function clean(){
        if (!Gate::allows('delete_backup') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        try {
            Artisan::call('backup:clean');
            // Artisan::call('backup:cleanall');
            return $this->dispatch('response', ['type' => 'success', 'message' => 'Backup cleaned successfully!']);
        } catch (Exception $e) {
            \Log::emergency("File:" . $e->getFile(). "Line:" . $e->getLine(). "Message:" . $e->getMessage());
            return $this->dispatch('response', ['type' => 'error', 'message' => $e->getMessage().' Something went wrong!']);
        }
    }
}
