<?php
namespace App\Livewire\Admin;


use App\Models\Module;
use App\Models\Permission;

use Livewire\Component;
use Livewire\WithPagination;
use App\Livewire\Traits\AuthorizesRequests;

use Illuminate\Support\Facades\Gate;

class PermissionComponent extends Component
{
    use WithPagination, AuthorizesRequests;
    public $search = '';
    public $perPage = 10;
    public $module, $name, $description,$modules;
    public $permissions = [];

    public $createModal = false;
    public $editModal = false;
    

    protected $listeners = ['createModal','editModal'];
    public function createModal(){
        $this->createModal = false;
    }
    public function editModal(){
        $this->editModal = false;
    }

    public function render()
    {
        if (!Gate::allows('create_permission') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }

        $query = Module::with('permissions');
        if($this->search){
            $query->where('name','like', '%'.$this->search.'%');
        }
        $collection = $query->paginate($this->perPage);
        return view('admin.permissions.index',compact('collection'));
    }

    public function create()
    {
        if (!Gate::allows('create_permission') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->pull();
        $this->resetErrorBag();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal']);
    }
    public function store()
    {
        if (!Gate::allows('create_permission') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->validate([
            'name'          =>  'required|alpha|max:50|unique:modules,name,NULL,id,deleted_at,NULL',
            'description'   => 'nullable|string|max:500|exclude',
        ]);
        $permissions = array('view_'.$this->name,'create_'.$this->name,'update_'.$this->name,'delete_'.$this->name,'status_'.$this->name);
        $module = Module::create(['name' => ucfirst($this->name)]);
        foreach($permissions as $permission){
            Permission::create(['name' => strtolower($permission), 'description' => $this->description, 'module_id' => $module->id]);
        }
        $this->dispatch('response', ['type' => 'success', 'message' => 'Permission has been created successfully!']);
        $this->dispatch('hideModal',['modal'=>'createModal']);
    }
    public function edit($id)
    {
        if (!Gate::allows('update_permission') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->editModal = true;
        $this->resetErrorBag();
        $this->module = Module::with('permissions')->find($id);
        $this->name = $this->module->name;
        $this->description= $this->module->description;
        $this->dispatch('showModal',['modal'=>'editModal']);
    }
    public function update($id)
    {
        if (!Gate::allows('update_permission') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $data = $this->validate([
            'name'              =>  'required|string|max:255',
            'description'       =>  'nullable|string|exclude',
        ]);
        $permissions = array('view_'.$this->name,'create_'.$this->name,'update_'.$this->name,'delete_'.$this->name,'status_'.$this->name);
        $this->module = Module::with('permissions')->find($id);
        $this->module->update(['name' => ucfirst($this->name)]);
        if(count($this->module->permissions)){
            foreach($this->module->permissions as $key => $p){
                $slug = $permissions[$key];
                Permission::where('module_id', $this->module->id)->whereId($p->id)->update(['name' => strtolower($slug), 'description' => $this->description]);
            }
        }else{
            foreach($permissions as $permission){
                Permission::create(['name' => strtolower($permission), 'description' => $this->description, 'module_id' => $this->module->id]);
            }
        }
        $this->dispatch('response', ['type' => 'success', 'message' => 'Permission has been updated successfully!']);
        $this->dispatch('hideModal',['modal'=>'editModal']);
    }

    public function delete($ids)
    {
        if (!Gate::allows('delete_permission') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = Module::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'Permission has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }
}
