<?php

namespace App\Livewire\Admin;

use App\Models\Place;
use App\Models\Language;
use App\Models\PlaceTranslation;

use App\Helpers\Helper;
use App\Livewire\Traits\AuthorizesRequests;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;

use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;

class PlaceComponent extends Component
{
    use WithPagination, WithFileUploads, AuthorizesRequests;
    public $search = '';
    public $perPage = 10;

    public $place;
    public $image;
    public $translations = [];
    public $languages;

    public $createModal = false;
    protected $listeners = ['createModal'];
    public function createModal(){
        $this->createModal = false;
    }
    public function create()
    {
        if (!Gate::allows('create_place') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->pull();
        $this->resetErrorBag();
        $this->place = new Place();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Create New Place']);
    }
    public function edit($id)
    {
        if (!Gate::allows('update_place') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->place = Place::with('translations')->findOrFail($id);
        // Initialize translations array
        $this->translations = [];

        foreach (Language::where('status', 'Active')->get() as $language) {
            $translation = $this->place->translations->where('language_id', $language->id)->first();

            $this->translations[$language->id] = [
                'name'        => $translation->name ?? '',
                'description' => $translation->description ?? '',
            ];
        }
        $this->resetErrorBag();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Update Place']);
    }
    public function save()
    {
        try {
            DB::transaction(function () {
                // Validate basic fields
                $validator = Validator::make([
                    'translations' => $this->translations,
                    'image'        => $this->image,
                ], [
                    'image'        => 'nullable|mimes:jpeg,png,jpg,webp|max:5120',
                ]);

                // Dynamic translation validation rules
                foreach (Language::where('status', 'Active')->get() as $language) {
                    $validator->addRules([
                        "translations.{$language->id}.name"   => 'required|string|max:100',
                        "translations.{$language->id}.description" => 'nullable|string|max:1000',
                    ]);
                }

                if ($validator->fails()) {
                    $this->setErrorBag($validator->errors());
                }

                $validated = $validator->validated();

                // Handle image upload if provided
                if ($this->image && $this->image->isValid()) {
                    // If editing and place already has image — delete it
                    if ($this->place->exists && $this->place->image) {
                        Helper::deleteFile($this->place->image);
                    }
                    // Save new image
                    $validated['image'] = Helper::saveFile('places', $this->image, $w=1200,$h=800);
                    $this->reset('image');
                } else {
                    // Keep existing image if editing, or null if creating
                    $validated['image'] = $this->place->image;
                }

                // Create or update the place
                $this->place->fill([
                    'image' => $validated['image'],
                ])->save();

                // Save translations
                foreach ($validated['translations'] as $language_id => $data) {
                    $this->place->translations()->updateOrCreate(
                        ['language_id' => $language_id],
                        [
                            'name'   => $data['name'],
                            'description' => $data['description'] ?? null,
                        ]
                    );
                }
            });

            $this->dispatch('response', [
                'type'    => 'success',
                'message' => 'Place has been ' . ($this->place->wasRecentlyCreated ? 'created' : 'updated') . ' successfully!'
            ]);
            $this->dispatch('hideModal', ['modal' => 'createModal']);
        } catch (\Throwable $e) {
            report($e);
            $this->dispatch('response', [
                'type'    => 'danger',
                'message' => 'Something went wrong. Please try again!'
            ]);
        }
    }

    public function delete($ids)
    {
        if (!Gate::allows('delete_place') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = Place::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'Place has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }
    public function render()
    {
        if (!Gate::allows('view_place') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        $query = Place::with('translations.language');
        if($this->search){
            $query->when($this->search, function ($q) {
                $q->whereHas('translations', function ($query) {
                    $query->where('name', 'like', '%' . $this->search . '%');
                });
            });
        }
        $collection = $query->paginate($this->perPage);
        $this->languages = Language::where('status', 'Active')->get();
        return view('admin.places.index',compact('collection'));
    }
}