<?php

namespace App\Livewire\Admin;

use App\Models\RoomService;
use App\Models\Language;

use Livewire\Component;
use Livewire\WithPagination;
use App\Livewire\Traits\AuthorizesRequests;

use Illuminate\Support\Str;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class RoomServiceComponent extends Component
{
    use WithPagination, AuthorizesRequests;

    public $search = '';
    public $perPage = 10;

    public $roomService,$code;
    public $translations = [];
    public $languages;

    public $editMode = false;
   
    
    public function edit($id)
    {
        if (!Gate::allows('update_roomservice') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->roomService = RoomService::with('translations')->findOrFail($id);
        // Initialize translations array
        $this->translations = [];

        foreach (Language::where('status', 'Active')->get() as $language) {
            $translation = $this->roomService->translations->where('language_id', $language->id)->first();
            $this->translations[$language->id] = [
                'name'        => $translation->name ?? '',
            ];
        }
        $this->resetErrorBag();
        $this->editMode = true;
    }
    public function save()
    {
        try {
            DB::transaction(function () {
                $this->roomService = $this->roomService ?? new RoomService();
                // Validate basic fields
                $validator = Validator::make([
                    'translations' => $this->translations,
                ],[
                    'translations' => 'required|array',
                ]);
                $languages = Language::where('status', 'Active')->get();
                // Dynamic translation validation rules
                foreach ($languages as $language) {
                    $validator->addRules([
                        "translations.{$language->id}.name"   => 'required|string|max:100',
                    ]);
                }

                if ($validator->fails()) {
                    $this->setErrorBag($validator->errors());
                }
                $validated = $validator->validated();
                $code = $this->translations[$languages[0]->id]['name'] ?? null;
                $existingCount = RoomService::where('code', 'LIKE', "{$code}%")->count();
                $validated['code'] = $existingCount ? "{$code}-{$existingCount}" : $code;
                // Create or update
                $this->roomService->fill([
                    'code'  => Str::slug($validated['code'], '-'),
                ])->save();
                // Save translations
                foreach ($validated['translations'] as $language_id => $data) {
                    $this->roomService->translations()->updateOrCreate(
                        ['language_id' => $language_id],
                        [
                            'name'   => $data['name'],
                        ]
                    );
                }
            });
            $this->editMode = false;
            $this->resetErrorBag();
            $this->dispatch('response', [
                'type'    => 'success',
                'message' => 'Room Service has been ' . ($this->roomService->wasRecentlyCreated ? 'created' : 'updated') . ' successfully!'
            ]);
            $this->pull();
            $this->dispatch('hideModal', ['modal' => 'createModal']);
        } catch (\Throwable $e) {
            report($e);
            $this->dispatch('response', [
                'type'    => 'danger',
                'message' => 'Something went wrong. Please try again!'
            ]);
        }
    }
    
    public function delete($ids)
    {
        if (!Gate::allows('delete_roomservice') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = RoomService::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'Room Service has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }
    public function render()
    {
        if (!Gate::allows('view_roomservice') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        $query = RoomService::query();
        if($this->search){
            $query->when($this->search, function ($q) {
                $q->whereHas('translations', function ($query) {
                    $query->where('name', 'like', '%' . $this->search . '%');
                });
            });
        }
        if($this->search){
            $query->where('code', 'like', '%' . $this->search. '%');
        }
        $collection = $query->orderBy('id','DESC')->paginate($this->perPage);
        $this->languages = Language::where('status', 'Active')->get();
        return view('admin.rooms.services',compact('collection'));
    }
}
