<?php

namespace App\Livewire\Admin;

use App\Models\User;
use App\Helpers\Helper;

use Livewire\Component;
use Livewire\WithFileUploads;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Gate;
use Illuminate\Contracts\Auth\StatefulGuard;
use Illuminate\Validation\ValidationException;
use Jenssegers\Agent\Agent;
use DB;
class UserAccountComponent extends Component
{
    use WithFileUploads;
    public $session = [];
    public User $user;
    public $name,$email,$phone,$password,$password_confirmation,$gender,$date_of_birth,$joining_date,$image,$change_password,$code;
    protected $listeners = ['createModal'];
    public $createModal = false;
    public function createModal(){
        $this->createModal = false;
    }
    public function render()
    {
        
        $this->user = auth()->user();
        $this->name = $this->user->name;
        $this->email= $this->user->email;
        $this->phone = $this->user->phone;
        $this->gender = $this->user->gender;
        $this->date_of_birth = !empty($this->user->date_of_birth) ? \Carbon\Carbon::parse($this->user->date_of_birth)->format('Y-m-d') : null;
        $this->joining_date = !empty($this->user->joining_date) ? \Carbon\Carbon::parse($this->user->joining_date)->format('Y-m-d') : null;
        $this->sessions  = $this->getSessions();
        return view('admin.users.user-account');
    }

    public function updateAccount()
    {
        $this->user = auth()->user();
        $data = $this->validate([
            'name'              =>  'required|string|max:255',
            'email'             =>  'required|string|email|max:50|unique:users,email,'.$this->user->id,
            'phone'             =>  'required|string|unique:users,phone,'.$this->user->id,
            'date_of_birth'     =>  'required|date_format:Y-m-d|before_or_equal:'.date('Y-m-d'),
            'image'             =>  'nullable|mimes:jpeg,png,jpg,webp|max:5120|exclude',
            'change_password'   =>  'nullable|boolean|exclude',
            'password'          =>  'required_if:change_password,==,true|nullable|string|min:8|confirmed|exclude',
        ]);
        if (!empty($this->password)) {
            $data['password'] = Hash::make($this->password); // Hash new password if provided
        }
        if($this->image && $this->image->isValid()){
            Helper::deleteFile($this->user->image);
            $data['image'] = Helper::saveFile('users',$this->image,$w=200,$h=200);
            $this->image->delete(); // Clean up the temporary file
        }
        $this->user->update($data);
        $this->reset('image');
        $this->dispatch('response', ['type' => 'success', 'message' => 'Your account has been updated successfully!']);
    }

    public function enable2FA()
    {
        auth()->user()->enableTwoFactorAuth();
        $this->dispatch('response', ['type' => 'success', 'message' => 'Two-factor authentication settings enabled successfully!']);
    }
    public function disable2FA()
    {
        auth()->user()->disableTwoFactorAuth();
        $this->dispatch('response', ['type' => 'success', 'message' => 'Two-factor authentication disabled successfully!']);
    }
    public function verify()
    {
        $data = $this->validate([
            'code'              =>  'required|string|max:255',
        ]);
        if(auth()->user()->verifyCode($this->code,null)){
            return $this->dispatch('response', ['type' => 'success', 'message' => 'Two-factor authentication enabled successfully!']);
        }   
        return $this->dispatch('response', ['type' => 'danger', 'message' => 'Invalid verification code!']);
    }

    public function regenerateRocoveryCodes(){
        auth()->user()->regenerateRocoveryCodes();
        return $this->dispatch('response', ['type' => 'success', 'message' => 'Rocovery Codes Generate successfully!']);
    }
    public function create() {

        $this->resetErrorBag();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal']);
    }
    public function logoutOtherBrowserSessions()
    {
        $data = $this->validate([
            'password'          =>  'required|string',
        ]);
        if (config('session.driver') !== 'database') {
            return;
        }
        $this->resetErrorBag();
        if (! Hash::check($this->password, Auth::user()->password)) {
            throw ValidationException::withMessages([
                'password' => [__('This password does not match our records.')],
            ]);
        }
        Auth::guard('web')->logoutOtherDevices($this->password);
        $this->deleteOtherSessionRecords();

        request()->session()->put([
            'password_hash_'.Auth::getDefaultDriver() => Auth::user()->getAuthPassword(),
        ]);
        $this->dispatch('response', ['type' => 'success', 'message' => 'Logged out from other browser sessions successfully!']);
        return $this->dispatch('hideModal',['modal'=>'createModal']);
    }
    protected function deleteOtherSessionRecords()
    {
        if (config('session.driver') !== 'database') {
            return;
        }

        DB::connection(config('session.connection'))->table(config('session.table', 'sessions'))
            ->where('user_id', Auth::user()->getAuthIdentifier())
            ->where('id', '!=', request()->session()->getId())
            ->delete();
    }
    public function getSessions()
    {
        if (config('session.driver') !== 'database') {
            return collect();
        }
        
        return collect(
            DB::connection(config('session.connection'))->table(config('session.table', 'sessions'))
                    ->where('user_id', Auth::user()->getAuthIdentifier())
                    ->orderBy('last_activity', 'desc')
                    ->get()
        )->map(function ($session) {
            return (object) [
                'agent' => Helper::createAgent($session),
                'ip_address' => $session->ip_address,
                'is_current_device' => $session->id === request()->session()->getId(),
                'last_active' => \Carbon\Carbon::createFromTimestamp($session->last_activity)->diffForHumans(),
            ];
        });
    }
}
