<?php

namespace App\Models;
use App\Traits\LogsActivity;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Crypt;
use PragmaRX\Google2FA\Google2FA;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use BaconQrCode\Writer;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasApiTokens, HasFactory, SoftDeletes, LogsActivity, Notifiable;
    protected $table = 'users';
    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
       'name',	
        'email',	
        'phone',	
        'password',	
        'gender',	
        'date_of_birth',	
        'joining_date',	
        'status',	
        'image',	
        'email_verified_at',
        'phone_verified_at',
        'verification_code',
        'verification_code_sent',
        'verification_code_sent_at',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'two_factor_confirmed_at'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'phone_verified_at'         => 'datetime',
            'two_factor_confirmed_at'   => 'datetime',
            'password' => 'hashed',
        ];
    }
    protected static function booted()
    {
        static::deleting(function ($model) {
            $model->roles()->sync([]);
            $model->permissions()->sync([]);
        });
    }
    public function sessions(): HasMany
    { 
        return $this->hasMany(UserLoginHistory::class)->orderBy('id','DESC');
    }
    public function roles(): BelongsToMany
    { 
        return $this->belongsToMany(Role::class, 'role_user');
    }

    public function permissions(): BelongsToMany
    {
        return $this->belongsToMany(Permission::class, 'permission_user');
    }

    // Check if user has a specific role
    public function hasRole($roleName)
    {
        return $this->roles->contains('name', $roleName);
    }

    public function hasPermissionTo($permission) {
        return $this->hasPermissionThroughRole($permission) || $this->hasPermission($permission);
    }

    public function hasPermissionThroughRole($permission): bool 
    {
        foreach ($permission->roles as $role){
            if($this->roles->contains($role)) {
                return true;
            }
        }
        return false;
    }
    protected function hasPermission($permission): bool
    {
        return (bool) $this->permissions->where('name', $permission->name)->first();
    }

    // Add these methods for 2FA
    public function enableTwoFactorAuth()
    {
        $google2fa = new Google2FA();
        
        $this->two_factor_secret = Crypt::encrypt(
            $google2fa->generateSecretKey()
        );
        
        $this->two_factor_recovery_codes = Crypt::encrypt(
            json_encode(collect()->times(8, function () {
                return $this->generateRecoveryCode();
            })->all())
        );
        $this->save();
    }

    public function disableTwoFactorAuth()
    {
        $this->two_factor_secret = null;
        $this->two_factor_recovery_codes = null;
        $this->two_factor_confirmed_at = null;
        $this->save();
    }
    public function regenerateRocoveryCodes()
    {
        if($this->two_factor_secret){
            $google2fa = new Google2FA();
            $this->two_factor_secret = Crypt::encrypt(
                $google2fa->generateSecretKey()
            );
            $this->two_factor_recovery_codes = Crypt::encrypt(
                json_encode(collect()->times(8, function () {
                    return $this->generateRecoveryCode();
                })->all())
            );
            $this->save();
        }
        return;
    }
    public function generateRecoveryCode()
    {
        return Str::random(10) . '-' . Str::random(10);
    }

    public function getTwoFactorQrCodeUrl()
    {
        return (new Google2FA)->getQRCodeUrl(
            config('app.name'),
            $this->email,
            Crypt::decrypt($this->two_factor_secret)
        );
    }
    public function twoFactorQrCodeSvg()
    {
        $google2fa = new Google2FA();
        $qrCodeUrl = $google2fa->getQRCodeUrl(
            config('app.name'),
            $this->email,
            Crypt::decrypt($this->two_factor_secret)
        );

        $svg = (new Writer(
            new ImageRenderer(
                new RendererStyle(150),
                new SvgImageBackEnd()
            )
        ))->writeString($qrCodeUrl);

        return $svg;
    }
    public function getRecoveryCodes()
    {
        return json_decode(Crypt::decrypt($this->two_factor_recovery_codes), true);
    }
    public function verifyCode($code = null,$recovery_code = null)
    {
        $google2fa = new Google2FA();
        if($code){
            if(empty($this->two_factor_secret)){
                throw ValidationException::withMessages([
                    'code' => ["Two factor authentication is not enabled."],
                ]);
            }
            $valid = $google2fa->verifyKey(Crypt::decrypt($this->two_factor_secret),$code);
            if ($valid) {
                session(['2fa_verified' => true]);
                $this->two_factor_confirmed_at = now();
                $this->save();
                return true;
            }else{
                throw ValidationException::withMessages([
                    'code' => ["The provided two factor authentication code was invalid."],
                ]);
            }
        }
        if ($recovery_code) {
            if(empty($this->two_factor_secret)){
                throw ValidationException::withMessages([
                    'recovery_code' => ["Two factor authentication is not enabled."],
                ]);
            }
            $recoveryCodes = json_decode(Crypt::decrypt($this->two_factor_recovery_codes), true);
            if (in_array($recovery_code, $recoveryCodes)) {
                // Remove the used recovery code
                $recoveryCodes = array_diff($recoveryCodes, [$recovery_code]);
                $this->two_factor_recovery_codes = Crypt::encrypt(json_encode($recoveryCodes));
                session(['2fa_verified' => true]);
                $this->two_factor_confirmed_at = now();
                $this->save();
                return true;
            }
            else{
                throw ValidationException::withMessages([
                    'recovery_code' => ["The provided recovery code was invalid."],
                ]);
            }
        }
    }

}
